/*
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "DM388_EVM.h"
#include "platform.h"
#include <stdlib.h>

extern diagTestTable_t gDiagTestTable[PLATFORM_TEST_COUNT];

/**
 *  \brief    Displays diagnostic test menu
 *
 *  \return   - None
 */
static void displayMenu(void)
{
	UINT16 count;
	platform_write("\n\n\r");
	platform_write("*******************************************************************\n\r");
	platform_write("                   DM388 CSK HW Diagnostic Tests                   \n\r");
	platform_write("*******************************************************************\n\r");
	platform_write("\n\nDiagnostic Tests       ");
	platform_write("\t\t\t Pass");
	platform_write("\t\t Fail\n\r");
	platform_write("-----------------------\t\t\t------\t\t------\n\r");

	for (count = 0; count < PLATFORM_TEST_COUNT; count++)
	{

		platform_write(" %2d - ",count);
		platform_write("%-30s", (const char*)gDiagTestTable[count].testName);
		platform_write("\t%3d", gDiagTestTable[count].pass);
		platform_write("\t\t%3d", gDiagTestTable[count].fail);
		platform_write("\n\r");
	}
	platform_write("\n\r");
	platform_write("  q - Quit\n\r");
	platform_write("\n\rEnter Desired Option:");
}

/**
 *  \brief    Executes all the diagnostic tests
 *
 *  \return   None
 */
static void run_all_tests(void)
{
	UINT16 index;

	platform_write("\n\n\rAuto Running All the Diagnostic Tests!\n\n\r");

	for (index = PLATFORM_TEST_START; index <= PLATFORM_AUTO_TEST_COUNT; index++)
	{
		if(gDiagTestTable[index].testFunction != NULL)
		{
			if(gDiagTestTable[index].testFunction(gDiagTestTable[index].args))
			{
				gDiagTestTable[index].fail++;
			}
			else
			{
				gDiagTestTable[index].pass++;
			}
		}
		else
		{
			platform_write("\n\nInvalid Test Handle for %s\n",
			               (const char*)gDiagTestTable[index].testName);
		}
	}

	platform_write("\nDiagnostic Tests Completed!!\n\n");
	displayMenu();
}

/**
 *  \brief    Diagnostic test main function
 *
 *  \return   Platform_STATUS
 */
INT32 diag_main (void)
{
	INT32 status;
	UINT8         exitTest;
	UINT8         inputErr;
	UINT32        diagInput;
	UINT8 		  testInput[2];

	exitTest = 0;
	inputErr = 0;

	/*Reading the input from the user*/

		while(!exitTest)
		{
			/* Display menu */
			displayMenu();

			testInput[0] = '\0';
			testInput[1] = '\0';

			status = platform_read(testInput, 3);
			if(status != -1)
			{
				/* Check user input */
				if((testInput[0] == 'q') || (testInput[0] == 'Q'))
				{
					exitTest = 1;
					platform_write("\n\nExiting the Diagnostic Tests!!\n\r");
					status = 0;
					break;
				}
				if((testInput[0] == 0xD) || (testInput[0] == 0x1B))
				{
					continue;
				}
				else
				{
					platform_write("\n\r");
					diagInput = strtoul((const char*)testInput, NULL, 10);
					switch(diagInput)
					{
						case PLATFORM_TEST_AUTO_ALL:
							while(1)
							{
							run_all_tests();
							}
							//break;
						default:
							if(diagInput < PLATFORM_TEST_COUNT)
							{
								if(gDiagTestTable[diagInput].testFunction != NULL)
								{
									if(gDiagTestTable[diagInput].testFunction(gDiagTestTable[diagInput].args))
									{
										gDiagTestTable[diagInput].fail++;
									}
									else
									{
										gDiagTestTable[diagInput].pass++;
									}
								}
								else
								{
									platform_write("\n\nInvalid Test Handle!!\n\r");
								}
							}
							else
							{
								inputErr = 1;
							}

							if(inputErr)
							{
								inputErr = 0;
								/* Invalid input. Get the desired option again and continue */
								platform_write("\n\n\rInvalid Input!\n\r");
								platform_write("Please Enter Valid Input\n\r");
								continue;
							}

							platform_write("\n\r");
							break;

					}
				}
			}
			else
			{
				platform_write("Platform Initialization Failed!\n");
			}
		}


	return 1;

}
